<?php
class Prosociate_Cron {
    /**
     * API key to prevent spam
     * @var int
     */
    private static $apiKey;

    /**
     * Only instance of the object
     *
     * @var null|self
     */
    private static $instance = null;

    /**
     * Number of seconds before update. Commonly 1 day (86400)
     */
    const UPDATE_SECONDS = 86400;

    function __construct() {
        // notes
        // 1.] No image handling to lessen processing and requests
        // 2.] Both requests and processing should be at minimum
        // 3.] Separate processing for non-variable and variable products might lessen the number of processes.
        add_action('wp_loaded', array($this, 'captureGet'));
    }

    /**
     * Singleton
     *
     * @return null|Prosociate_Cron
     */
    public static function getInstance() {
        if(self::$instance === null)
            self::$instance = new self;

        return self::$instance;
    }

    /**
     * Set the api key
     */
    public function getApi() {
        // Check if we dont have the api key yet
        if(self::$apiKey === null || self::$apiKey === '')
            self::$apiKey = get_option('prossociate_settings-dm-cron-api-key', '');
    }

    /**
     * Capture the cron starter
     */
    public function captureGet() {
        // Get api
        $this->getApi();
        // Check if we have the proper initializers and api key
        if(isset($_GET['proscron']) && $_GET['proscron'] == self::$apiKey) {
            update_option('prossociate_settings-hide-cron', 'hide');
            $this->startCron();
        }
    }

    /**
     * Start the cron process
     */
    private function startCron() {
        $products = $this->getProducts();

        // Prevent further actions if no products need to be updated
        if(empty($products)) {
            // If no products needs update. Do the subscription cron
            new ProsociateCronSubscription();
        } else {
            // Get if we need to delete
            $deleteUnavailable = get_option('prossociate_settings-dm-pros-prod-avail', false);

            // Process each product
            foreach($products as $product) {
                // Cache the product ID
                $dmProductId = $product->ID;

                // Get asin of product
                $asin = get_post_meta($product->ID, '_pros_ASIN', true);

                // Get data of the asin
                $data = $this->getData($asin);

                // Check if we got the data
                if($data === false)
                    break;

                // Check for the availability of the product
                $available = $this->checkAvailability($data);

                // If available update the product
                if($available) {
                    // Update product meta
                    $this->updateProduct($product->ID, $data);
                    // Make it have "stock"
                    update_post_meta($product->ID, '_stock_status', 'instock');
                } else {
                    $byPassExternal = false;
                    // Check if the product is external
                    $isExternal = get_post_meta( $product->ID, '_dmaffiliate', true );

                    // If the product is external and it has variationsummary then it has stocks
                    if(!empty($isExternal) && isset($data->VariationSummary))
                        $byPassExternal = true;

                    // Check if variable product
                    if(isset($data->Variations) || $byPassExternal) {
                        // Check for variations without attribute value
                        if($this->checkVariationAttributesValue($data, $product->ID)) {
                            // Update the parent product
                            $this->updateProduct($product->ID, $data, true);

                            if( function_exists('get_product') ){
                                $product = get_product($product->ID);
                                if( $product->is_type( 'external' ) ){

                                } else {
                                    // Process variable products here
                                    $items = $this->getVariations($data->Variations);
                                    // Update the variations
                                    $this->updateVariations($items);
                                }
                            }

                            // If variation make it on stock
                            update_post_meta($product->ID, '_stock_status', 'instock');
                            // Process variable products here
                            $items = $this->getVariations($data->Variations);
                            // Update the variations
                            $this->updateVariations($items);
                        }
                    } else {
                        // Check if we need to delete
                        if($deleteUnavailable == 'remove') {
                            // Check if there are variations and delete it
                            $productVariations = $this->getProductVariations($product->ID);
                            if(!empty($productVariations)) {
                                foreach($productVariations as $k) {
                                    // Delete the images
                                    $this->deleteImages($k->ID);
                                    // Delete the product
                                    $this->deleteProduct($k->ID);
                                }
                            }

                            // Delete the main product
                            // Delete the images
                            $this->deleteImages($product->ID);
                            // Delete the product
                            $this->deleteProduct($product->ID);
                        } else {
                            // Update product meta
                            $this->updateProduct($product->ID, $data);
                            // Make out of stock
                            update_post_meta($product->ID, '_stock_status', 'outofstock');
                        }
                    }

                }

                // Update the update time
                update_post_meta($dmProductId, '_pros_last_update_time', time());
            }
        }

        // Update Pros Cron Job timer
        update_option('prosLastCronTime', time());
    }

    /**
     * Get products that needs to be updated
     * @return array
     */
    public function getProducts() {
        // Get products that are not updated within 24 hours
        $products = new WP_Query(
            array(
                'post_type' => 'product',
                'post_status' => 'publish',
                'posts_per_page' => 200, // Assuming that we can only process less than 100 products per request
                'meta_query' => array(
                    array(
                        'key' => '_pros_last_update_time',
                        'value' => time() - self::UPDATE_SECONDS,
                        'compare' => '<='
                    )
                )
            )
        );

        // Make sure we reset
        wp_reset_postdata();

        return $products->posts;
    }

    /**
     * Get product data from amazon
     * @param $asin
     * @return object|false
     */
    private function getData($asin) {
        // Instantiate the lib
        $amazonEcs = new AmazonECS(AWS_API_KEY, AWS_API_SECRET_KEY, AWS_COUNTRY, AWS_ASSOCIATE_TAG);

        // Include the merchant ID
        $amazonEcs->optionalParameters(array('MerchantId' => 'All'));

        // Try to get the data
        try {
            $response = $amazonEcs->responseGroup('Large,Variations,OfferFull,VariationOffers')->lookup($asin);
            $data = $response->Items->Item;
        } catch(Exception $exception) {
            $data = false;
        }

        return $data;
    }

    /**
     * Check if variation has attribute values
     * @param $data
     * @param int $post_id
     * @return bool
     */
    private function checkVariationAttributesValue($data, $post_id) {
        $dmValuePresent = true;
        // Check if array
        if(is_array($data->Variations->Item)) {
            // Check if variation attributes is an array
            if(is_array($data->Variations->Item[0]->VariationAttributes->VariationAttribute)) {
                // Check if value is present
                if(!isset($data->Variations->Item[0]->VariationAttributes->VariationAttribute[0]->Value))
                    $dmValuePresent = false;
            } else {
                if(!isset($data->Variations->Item[0]->VariationAttributes->VariationAttribute->Value))
                    $dmValuePresent = false;
            }
        } else {
            // Check if variation attributes is an array
            if(is_array($data->Variations->Item->VariationAttributes->VariationAttribute)) {
                if(!isset($data->Variations->Item->VariationAttributes->VariationAttribute[0]->Value))
                    $dmValuePresent = false;
            } else {
                if(!isset($data->Variations->Item->VariationAttributes->VariationAttribute->Value))
                    $dmValuePresent = false;
            }
        }

        // If Value isn't present dont import the product. It will create broken variations
        if(!$dmValuePresent) {
            // Delete the product
            // Check if there are variations and delete it
            $productVariations = $this->getProductVariations($post_id);
            if(!empty($productVariations)) {
                foreach($productVariations as $k) {
                    // Delete the images
                    $this->deleteImages($k->ID);
                    // Delete the product
                    $this->deleteProduct($k->ID);
                }
            }

            // Delete the main product
            // Delete the images
            $this->deleteImages($post_id);
            // Delete the product
            $this->deleteProduct($post_id);
        }

        return $dmValuePresent;
    }

    /**
     * Update the prices of the product
     * @param int $productId
     * @param object $data
     * @param bool $isVariation
     */
    private function updatePrices($productId, $data, $isVariation = false) {
        // Check if array
        if(is_array($data->Offers->Offer)) {
            foreach($data->Offers->Offer as $offer) {
                // Check if there's no offer listing
                if(!isset($offer->OfferListing->OfferListingId)) {
                    continue;
                } else {
                    $finalOffer = $offer->OfferListing->OfferListingId;
                    $finalPrice = $offer->OfferListing->Price->FormattedPrice;
                    $finalAmount = $offer->OfferListing->Price->Amount;
                    // Check if sale price is given
                    if(isset($offer->OfferListing->SalePrice)) {
                        $finalSalePrice = $this->reformat_prices($this->remove_currency_symbols($offer->OfferListing->SalePrice->FormattedPrice));
                        $finalSaleAmount = $offer->OfferListing->SalePrice->Amount;
                    } else {
                        $finalSaleAmount = 0;
                    }
                    break;
                }
            }
        } elseif($data->Offers->TotalOffers === 0 && !isset($data->Variations) && isset($data->VariationSummary)) {
            // This is for products without offers and variations listings
            // Example http://www.amazon.com/Sherri-Hill-21002/dp/
            if(isset($data->VariationSummary->LowestPrice)) {
                $finalPrice = $data->VariationSummary->LowestPrice->FormattedPrice;
                $finalAmount = $data->VariationSummary->LowestPrice->Amount;
            }
        } else {
            // For non-array
            // Check if offer listing exists
            if(isset($data->Offers->Offer->OfferListing->OfferListingId)) {
                $finalOffer = $data->Offers->Offer->OfferListing->OfferListingId;
                $finalPrice = $data->Offers->Offer->OfferListing->Price->FormattedPrice;
                $finalAmount = $data->Offers->Offer->OfferListing->Price->Amount;
                // Check if sale price is given
                if(isset($data->Offers->Offer->OfferListing->SalePrice)) {
                    $finalSalePrice = $this->reformat_prices($this->remove_currency_symbols($data->Offers->Offer->OfferListing->SalePrice->FormattedPrice));
                    $finalSaleAmount = $data->Offers->Offer->OfferListing->SalePrice->Amount;
                } else {
                    $finalSaleAmount = 0;
                }
            }
        }

        // If variation
        if($isVariation) {
            // Get the price of the very first variation
            if(is_array($data->Variations->Item)) {
                $finalPrice = $data->Variations->Item[0]->Offers->Offer->OfferListing->Price->FormattedPrice;
            } else {
                $finalPrice = $data->Variations->Item->Offers->Offer->OfferListing->Price->FormattedPrice;
            }

            // Prices for external variable products
            // Get the sale price
            $finalSaleAmount = 0;
            if(isset($data->VariationSummary->LowestSalePrice)) {
                $finalSalePrice = $this->reformat_prices($this->remove_currency_symbols($data->VariationSummary->LowestSalePrice->FormattedPrice));
                $finalSaleAmount = $data->VariationSummary->LowestSalePrice->Amount;
            }
        }

        $isExternal = get_post_meta( $productId, '_dmaffiliate', true );

        if( !empty( $isExternal ) && isset($data->Variations) ) {
            // Get the price of the very first variation
            if(is_array($data->Variations->Item)) {
                $finalPrice = $data->Variations->Item[0]->Offers->Offer->OfferListing->Price->FormattedPrice;
            } else {
                $finalPrice = $data->Variations->Item->Offers->Offer->OfferListing->Price->FormattedPrice;
            }

            // Prices for external variable products
            // Get the sale price
            $finalSaleAmount = 0;
            if(isset($data->VariationSummary->LowestSalePrice)) {
                $finalSalePrice = $this->reformat_prices($this->remove_currency_symbols($data->VariationSummary->LowestSalePrice->FormattedPrice));
                $finalSaleAmount = $data->VariationSummary->LowestSalePrice->Amount;
            }

        }

        // Handle the price
        $finalProcessedPrice = $this->reformat_prices($this->remove_currency_symbols($finalPrice));

        // Add the offer id and price
        update_post_meta($productId, '_dmpros_offerid', $finalOffer);
        update_post_meta($productId, '_price', $finalProcessedPrice);
        update_post_meta($productId, '_regular_price', $finalProcessedPrice);

        // Handle prices with Too low to display
        if($finalPrice === 'Too low to display') {
            update_post_meta($productId, '_price', '0');
            update_post_meta($productId, '_regular_price', '0');
            update_post_meta($productId, '_filterTooLowPrice', 'true');
        } elseif($finalSaleAmount > 0) {  // Handle the regular / sale price
            update_post_meta($productId, '_regular_price',$finalProcessedPrice);
            update_post_meta($productId, '_sale_price', $finalSalePrice);
            update_post_meta($productId, '_price', $finalSalePrice);
        }
    }

    /**
     * Delete all the price metas
     * @param int $productId
     */
    private function deletePrices($productId) {
        delete_post_meta($productId, '_regular_price');
        delete_post_meta($productId, '_sale_price');
        delete_post_meta($productId, '_price');
    }

    /**
     * Reformat the price
     * @param $price
     * @return mixed
     */
    private function reformat_prices($price) {
        switch( AWS_COUNTRY ) {
            // Germany
            case 'de':
                $formatPrice = $this->reformat_price_de($price);
                break;
            // France
            case 'fr':
                $formatPrice = $this->reformat_price_de($price);
                break;
            // Spain
            case 'es':
                $formatPrice = $this->reformat_price_de($price);
                break;
            // Italy
            case 'it':
                $formatPrice = $this->reformat_price_de($price);
                break;
            default:
                $formatPrice = str_replace(',', '', $price);
                break;
        }

        return $formatPrice;
    }

    /**
     * @param $price
     * @return string
     */
    private function reformat_price_de($price) {
        // Convert the string to array
        $priceArray = str_split($price);
        foreach ($priceArray as $k => $v) {
            // Check if a period
            if ($v == '.') {
                // Convert the period to comma
                $priceArray[$k] = '';
            } elseif ($v == ',') {
                // Convert comma to period
                $priceArray[$k] = '.';
            }
        }
        // Convert the array to a string
        $formatPrice = implode('', $priceArray);

        return $formatPrice;
    }

    /**
     * Remove the currency symbol
     * @param $x
     * @return mixed
     */
    private function remove_currency_symbols($x) {
        $x = preg_replace('/[^0-9-.,]/', '', $x);

        // strip spaces, just in case
        $x = str_replace(" ", "", $x);

        return $x;
    }

    /**
     * Update the fields of the products
     * @param int $productId
     * @param object $data
     */
    private function updateCustomFields($productId, $data) {
        if(isset($data->ItemAttributes)) {
            update_post_meta($productId, '_pros_ItemAttributes', serialize($data->ItemAttributes));
        }
        if(isset($data->Offers)) {
            update_post_meta($productId, '_pros_Offers', serialize($data->Offers));
        }
        if(isset($data->OfferSummary)) {
            update_post_meta($productId, '_pros_OfferSummary', serialize($data->OfferSummary));
        }
        if(isset($data->SimilarProducts)) {
            update_post_meta($productId, '_pros_SimilarProducts', serialize($data->SimilarProducts));
        }
        if(isset($data->Accessories)) {
            update_post_meta($productId, '_pros_Accessories', serialize($data->Accessories));
        }
        if(isset($data->ASIN)) {
            update_post_meta($productId, '_pros_ASIN', $data->ASIN);
            // Add sku
            update_post_meta($productId, '_sku', $data->ASIN);
        }

        if(isset($data->ParentASIN)) {
            update_post_meta($productId, '_pros_ParentASIN', $data->ParentASIN);
        }
        if(isset($data->DetailPageURL)) {
            update_post_meta($productId, '_pros_DetailPageURL', $data->DetailPageURL);
        }
        if(isset($data->CustomerReviews)) {
            update_post_meta($productId, '_pros_CustomerReviews', serialize($data->CustomerReviews));
        }
        if(isset($data->EditorialReviews)) {
            update_post_meta($productId, '_pros_EditorialReviews', serialize($data->EditorialReviews));
        }
        if(isset($data->VariationSummary)) {
            update_post_meta($productId, '_pros_VariationSummary', serialize($data->VariationSummary));
        }
        if(isset($data->Variations->VariationDimensions)) {
            update_post_meta($productId, '_pros_VariationDimensions', serialize($data->Variations->VariationDimensions));
        }

        if(isset($data->DetailPageURL)) {
            // Set the product url
            $this->wpWizardCloakIntegration($productId, $data->DetailPageURL);
        }

        if(isset($data->Variations->TotalVariations)) {
            if ($data->Variations->TotalVariations > 0) {
                if (count($data->Variations->Item) == 1) {
                    update_post_meta($productId, '_pros_FirstVariation', serialize($data->Variations->Item));
                } else {
                    update_post_meta($productId, '_pros_FirstVariation', serialize($data->Variations->Item[0]));
                }
            }
        }
    }

    /**
     * Process all the meta fields of product
     * @param int $productId
     * @param object $data
     * @param bool $isVariation
     */
    private function updateProduct($productId, $data, $isVariation = false) {
        // Delete current prices
        $this->deletePrices($productId);
        // Update the price
        $this->updatePrices($productId, $data, $isVariation);
        // Update the custom fields
        $this->updateCustomFields($productId, $data);
    }

    /**
     * Delete associated images for a product
     * @param int $productId
     */
    private function deleteImages($productId) {
        // Get main images
        $args = array(
            'post_parent' => (int)$productId,
            'post_type' => 'attachment',
            'post_status' => 'any'
        );
        $images = get_children($args);

        // Delete images
        foreach($images as $image) {
            wp_delete_attachment($image->ID, true);
        }
    }

    /**
     * Delete a product
     * @param int $productId
     */
    private function deleteProduct($productId) {
        // Delete
        wp_delete_post($productId, true);
    }

    /**
     * Check if the product is available
     * @param object $data
     * @return bool
     */
    private function checkAvailability($data) {
        $available = false;

        // Check for availability
        if(isset($data->Offers->TotalOffers) && $data->Offers->TotalOffers > 0) {
            $available = true;
        }

        return $available;
    }


    /**
     * Retrieve the product_variations
     * @param int $postId
     * @return array|bool
     */
    private function getProductVariations($postId) {
        $children = get_children(array(
            'post_parent' => $postId,
            'post_type' => 'product_variation',
            'numberposts' => -1,
            'post_status' => 'any'
        ));

        return $children;
    }

    /**
     * Get the variations from the data
     * @param object $varData
     * @return array
     */
    private function getVariations($varData) {
        // Check if we have variations
        if(isset($varData->Item) && !empty($varData->Item)) {
            return $varData->Item;
        } else {
            return array();
        }
    }

    /**
     * Update the variations of a product
     * @param array $variations
     */
    private function updateVariations($variations) {
        // Check if on array
        if(is_array($variations)) {
            // Loop through all variations
            foreach($variations as $var) {
                // Get the post id of variation
                $postId = $this->getProductByAsin($var->ASIN);
                // Check if we dont any variations
                if($postId === 0) {
                    // TODO create a new variation
                } else {
                    // Check if we need to update the product
                    $update = $this->isLessThanADay($postId);
                    if($update) {
                        // Delete prices
                        $this->deletePrices($postId);
                        // Update prices
                        $this->updatePrices($postId, $var);
                        // Update the last cron time meta
                        update_post_meta($postId, '_pros_last_update_time', time());
                        // Insert sku
                        update_post_meta($postId, '_sku', $var->ASIN);
                    }
                }
            }
        } else {
            // Not array
            // Get the post id of variation
            $postId = $this->getProductByAsin($variations->ASIN);
            // Check if we dont any variations
            if($postId === 0) {
                // TODO create a new variation
            } else {
                // Check if we need to update the product
                $update = $this->isLessThanADay($postId);
                if($update) {
                    // Delete prices
                    $this->deletePrices($postId);
                    // Update prices
                    $this->updatePrices($postId, $variations);
                    // Update the last cron time meta
                    update_post_meta($postId, '_pros_last_update_time', time());
                    // Insert sku
                    update_post_meta($postId, '_sku', $variations->ASIN);
                }
            }
        }

    }

    /**
     * Get post id of a product by asin
     * @param string $asin
     * @return int
     */
    private function getProductByAsin($asin) {
        // By default we dont have any product ID
        $productId = 0;

        // Get product
        $products = get_posts(array(
            'post_status' => array('publish', 'draft'),
            'post_type' => array('product', 'product_variation'),
            'meta_query' => array(
                array(
                    'key' => '_pros_ASIN',
                    'value' => $asin,
                    'compare' => '='
                )
            )
        ));

        // Check if we got any product
        if(!empty($products)) {
            // Only get the first post ID
            $productId = $products[0]->ID;
        }

        return $productId;
    }

    /**
     * Check if we need to update a product by determining it's last update time
     * @param int $postId
     * @return bool
     */
    private function isLessThanADay($postId) {
        // Update by default
        $update = true;

        // Get last update time
        $lastUpdateTime = get_post_meta($postId, '_pros_last_update_time', true);

        // Check if we get last update time
        if(!empty($lastUpdateTime)) {
            // Check if the last update was within 24 hours
            if((int)$lastUpdateTime >= (time() - self::UPDATE_SECONDS)) {
                $update = false;
            }
        }

        return $update;
    }

    /**
     * Get the right term slug
     * @param string $value Taxonomy value e.g 'pa_color', 'pa_size'
     * @param string $key Variation value e.g Black, Green
     * @return string
     */
    private function getVariationMeta($value, $key) {
        global $wpdb, $woocommerce;
        $metaValue = $woocommerce->attribute_taxonomy_name(strtolower($value));

        $sql = "SELECT {$wpdb->terms}.slug FROM {$wpdb->terms} INNER JOIN {$wpdb->term_taxonomy} ON {$wpdb->terms}.term_id = {$wpdb->term_taxonomy}.term_id
WHERE {$wpdb->terms}.name =  '{$key}'
AND {$wpdb->term_taxonomy}.taxonomy = '{$metaValue}'";
        $result = $wpdb->get_var($sql);

        // If no result was found
        if($result === null)
            return '';
        else
            return $result;
    }

    /**
     * Add WP Wizard Cloak integration
     * @param int $postId
     * @param string $buyUrl
     */
    private function wpWizardCloakIntegration($postId, $buyUrl) {
        if(empty($buyUrl))
            return;

        // Check if WP Wizard Cloak is activated
        if (in_array('wpwizardcloak/plugin.php', apply_filters('active_plugins', get_option('active_plugins')))) {
            // Save the original url on another meta
            update_post_meta($postId, '_orig_buy_url', $buyUrl);
            // Cloak the link
            $cloak = new Prosociate_WPWizardCloak($postId, $buyUrl);
            // Get the cloaked link
            if(!empty($cloak->cloakedLink)) {
                $cloakLink = get_bloginfo('wpurl') . '?cloaked=' . $cloak->cloakedLink;
                update_post_meta($postId, '_product_url', $cloakLink);
            }
        } else {
            update_post_meta($postId, '_product_url', $buyUrl);
        }
    }
}
Prosociate_Cron::getInstance();